//=============================================================================
// SoR_VoiceSoundManager_MZ.js
// SoR License (C) 2020 蒼竜, REQUIRED User Registration on Dragon Cave
// http://dragonflare.blue/dcave/license.php
// ----------------------------------------------------------------------------
// Latest version v1.10 (2024/03/10)
//=============================================================================
/*:ja
@plugindesc ＜ボイスサウンド機能＞ v1.10
@author 蒼竜
@target MZ
@url https://dragonflare.blue/dcave/
@help 「メッセージの表示」コマンドに連動する
ボイスサウンド(汎用のテキストサウンド用途以外の音声ファイル)の
独立した再生設定・管理機能を実装します。

対応プラグイン：
9.「おしゃべりポップアップウィンドウ」(v1.32以上)
16.「メッセージウィンドウ拡張」(v1.53以上)
未実装：
18.「メッセージバックログ」

@param DefaultVolume_Option
@desc オプション内ボイス音声のボリューム初期値 (default: 100)
@type number
@default 100
@max 100

@param OptionText_VoiceVolume
@desc オプション内ボリューム設定項目の名称テキスト
@default Voice 音量
@type string

@param GlobalVolume_Ratio
@desc ゲーム全体に渡る音声再生ボリューム倍率。個別設定の音量に指定割合％がけする (default: 100)
@type number
@default 100
@max 200

@command VoiceAllocation
@text サウンド割り当て[ボイスサウンド機能]
@desc 同一イベント内のメッセージ表示に合わせて再生される音声データを予約します。
@arg sounds
@type struct<SEDATA>[]
@desc 再生予約を行うサウンドデータ
@default ["{\"name\":\"\",\"volume\":\"100\",\"pitch\":\"100\",\"pan\":\"0\"}"]
@command VoiceAllocationG
@text サウンド割り当て(Gab)[ボイスサウンド機能]
@desc 「おしゃべりポップアップ」メッセージ表示に連動して再生される音声データを予約します。
@arg sounds
@type struct<SEDATA>[]
@desc 再生予約を行うサウンドデータ
@default ["{\"name\":\"\",\"volume\":\"100\",\"pitch\":\"100\",\"pan\":\"0\"}"]
*/
/*
@plugindesc <Voice Sound Manager> v1.10
@author Soryu
@target MZ
@url https://dragonflare.blue/dcave/index_e.php
@help This plugin presents a mechanism of independent sound manager
for voices associated with message texts on the game.

Collaborating plugins:
9. SoR_GabWindow (v1.32 or upper)
15. SoR_MessageExtension (v1.53 or upper)

Not implemented currently:
18. SoR_MessageBackLog

@param DefaultVolume_Option
@desc Default voice volume in the option (default: 100)
@type number
@default 100
@max 100

@param OptionText_VoiceVolume
@desc A text of voice volume configuration in the option
@default Voice Volume
@type string

@param GlobalVolume_Ratio
@desc Volume rate of voice sounds in the entire game. X% is multiplied to every voice play setting. (default: 100)
@type number
@default 100
@max 100


@command VoiceAllocation
@text Voice Allocation [Voice Sound Manager]
@desc Reserve voice sounds which are associated with message text processes in the event.
@arg sounds
@type struct<SEDATAE>[]
@desc Queue array of Sound data which will be played 
@default ["{\"name\":\"\",\"volume\":\"100\",\"pitch\":\"100\",\"pan\":\"0\"}"]

@command VoiceAllocationG
@text Voice Allocation (Gab) [Voice Sound Manager]
@desc Reserve voice sounds which are associated with SoR_GabWindow message texts.
@arg sounds
@type struct<SEDATA>[]
@desc Queue array of Sound data which will be played 
@default ["{\"name\":\"\",\"volume\":\"100\",\"pitch\":\"100\",\"pan\":\"0\"}"]
*/
/*~struct~SEDATA:
@type string
@param name
@dir audio/voice/
@type file
@desc 効果音
@param volume
@desc 音量 [0...100]
@type number
@default 100
@min 0
@max 100
@param pitch
@desc ピッチ [50...150]
@type number
@default 100
@min 50
@max 150
@param pan
@desc パン(位相) [-50...50]
@type number
@default 0
@min -50
@max 50
*/
/*~struct~SEDATAE:
@type string
@param name
@dir audio/voice/
@type file
@desc SE File
@param volume
@desc Voulme [0...100]
@type number
@default 100
@min 0
@max 100
@param pitch
@desc Pitch [50...150]
@type number
@default 100
@min 50
@max 150
@param pan
@desc Pan [-50...50]
@type number
@default 0
@min -50
@max 50
*/
(function() {
const pluginName = "SoR_VoiceSoundManager_MZ";
const Param = PluginManager.parameters(pluginName);

const G_volumeRate = Number(Param['GlobalVolume_Ratio']) * 0.01;
const OptionText_VoiceVolume = String(Param['OptionText_VoiceVolume']); 
const DefaultOptVolume = Number(Param['DefaultVolume_Option']); 


PluginManager.registerCommand(pluginName, "VoiceAllocation", args => {
    const arr = convertJsonParams(args.sounds);
    $game_VSounds.addQsound(arr);
});

PluginManager.registerCommand(pluginName, "VoiceAllocationG", args => {
    const arr = convertJsonParams(args.sounds);
    $game_VSounds.addQsoundG(arr);
});

function convertJsonParams(param) {
    if (param == undefined) return [];
    let arr = [];
        JSON.parse(param).map(function(param) {
        const obj = JSON.parse(param);
            obj.volume = Number(obj.volume);
            obj.pan = Number(obj.pan);
            obj.pitch = Number(obj.pitch);
            arr.push(obj);
        });
    return arr;
}

const SoR_VSM_DM_createGameObjects = DataManager.createGameObjects;
DataManager.createGameObjects = function() {
    SoR_VSM_DM_createGameObjects.call(this);
    $game_VSounds = new Game_VSoundSoR();
}

///////////////////////////////////////////////////////////////////////////////////
const SoR_VSM_WM_startMessage = Window_Message.prototype.startMessage;
Window_Message.prototype.startMessage = function() {
    SoR_VSM_WM_startMessage.call(this);
    $game_VSounds.processVoiceSounds();
}

const SoR_VSM_WM_terminateMessage = Window_Message.prototype.terminateMessage;
Window_Message.prototype.terminateMessage = function() {
   SoR_VSM_WM_terminateMessage.call(this);
   AudioManager.stopVoiceSounds();
}


const SoR_VSM_SM_goto = SceneManager.goto;
SceneManager.goto = function(sceneClass) {
    SoR_VSM_SM_goto.call(this, sceneClass);
    if(typeof $game_VSounds != "undefined") $game_VSounds.suspendVoiceSounds();
}

///////////////////////////////////////////////////////////////////////////////////

Object.defineProperty(ConfigManager, "SoRVoiceSEVolume", {
    get: function() {
        return AudioManager._voiceVolume;
    },
    set: function(value) {
        AudioManager._voiceVolume = value;
    },
    configurable: true
});


const SoR_VSM_CM_makeData = ConfigManager.makeData;
ConfigManager.makeData = function() {
    const config = SoR_VSM_CM_makeData.call(this);
    config.SoRVoiceSEVolume = this.SoRVoiceSEVolume;
    return config;
}

const SoR_VSM_CM_applyData = ConfigManager.applyData;
ConfigManager.applyData = function(config) {
    SoR_VSM_CM_applyData.call(this,...arguments);
    this.SoRVoiceSEVolume = this.readVolume(config, "SoRVoiceSEVolume");
}
 
const SoR_VSM_WO_makeCommandList = Window_Options.prototype.makeCommandList;
Window_Options.prototype.makeCommandList = function() {
    SoR_VSM_WO_makeCommandList.call(this);
    this.addSoR_VoiceSEVolOption();
}

Window_Options.prototype.addSoR_VoiceSEVolOption = function() {
    //manual addCommand
    const idx = this._list.findIndex(x=> x.symbol.includes("Volume"));
    let addidx = idx;
    for(let i=idx+1; i< this._list.length; i++){
        if(!this._list[i].symbol.includes("Volume")){
            addidx = i;
            break;
        }
        if(i+1==this._list.length){
            addidx = i+1;
            break;
        }
    }

    this._list.splice(addidx,0, { name: OptionText_VoiceVolume, symbol: "SoRVoiceSEVolume", enabled: true, ext: null });
}

const SoR_ME_SO_maxCommands = Scene_Options.prototype.maxCommands;
Scene_Options.prototype.maxCommands = function() {
    // Increase this value when adding option items.
    const defaults = SoR_ME_SO_maxCommands.call(this);
    return 1 + defaults;
}



///////////////////////////////////////////////////////////////////////////////////

AudioManager._voiceVolume = DefaultOptVolume;
AudioManager._voiceBuffers = []; 
AudioManager._staticBuffersVS = [];
AudioManager._voiceBuffersG = []; 
AudioManager._staticBuffersVSG = [];

Object.defineProperty(AudioManager, "SoRVoiceSEVolume", {
    get: function() {
        return this._voiceVolume;
    },
    set: function(value) {
        this._voiceVolume = value;
    },
    configurable: true
});



AudioManager.playVoiceSound = function(se) {
    if (se.name) {
        // [Note] Do not play the same sound in the same frame.
        const latestBuffers = this._voiceBuffers.filter(
            buffer => buffer.frameCount === Graphics.frameCount
        );
        if (latestBuffers.find(buffer => buffer.name === se.name)) {
            return;
        }
        const buffer = this.createBuffer("voice/", se.name);
        this.updateVSParameters(buffer, se);
        buffer.play(false);
        this._voiceBuffers.push(buffer);
        this.cleanupVS();
    }
}

AudioManager.updateVSParameters = function(buffer, se) {
    this.updateBufferParameters(buffer, this._voiceVolume * G_volumeRate, se);
}

AudioManager.cleanupVS = function() {
    for (const buffer of this._voiceBuffers) {
        if (!buffer.isPlaying()) buffer.destroy();
    }
    this._voiceBuffers = this._voiceBuffers.filter(buffer => buffer.isPlaying());
}

AudioManager.stopVoiceSounds = function(allQclear) {
    for (const buffer of this._voiceBuffers) {
        buffer.destroy();
    }
    this._voiceBuffers = [];
    if(allQclear) $game_VSounds.allQclear();
}

AudioManager.playStaticVS = function(se) {
    if (se.name) {
        this.loadStaticVS(se);
        for (const buffer of this._staticBuffersVS) {
            if (buffer.name === se.name) {
                buffer.stop();
                this.updateSeParameters(buffer, se);
                buffer.play(false);
                break;
            }
        }
    }
}

AudioManager.loadStaticVS = function(se) {
    if (se.name && !this.isStaticVS(se)) {
        const buffer = this.createBuffer("voice/", se.name);
        this._staticBuffersVS.push(buffer);
    }
}

AudioManager.isStaticVS = function(se) {
    for (const buffer of this._staticBuffersVS) {
        if (buffer.name === se.name) {
            return true;
        }
    }
    return false;
}

//////////////////////////////////////////////////////////////////////////////////

AudioManager.playVoiceSoundG = function(se) {
    if (se.name) {
        // [Note] Do not play the same sound in the same frame.
        const latestBuffers = this._voiceBuffersG.filter(
            buffer => buffer.frameCount === Graphics.frameCount
        );
        if (latestBuffers.find(buffer => buffer.name === se.name)) {
            return;
        }
        const buffer = this.createBuffer("voice/", se.name);
        this.updateVSParameters(buffer, se);
        buffer.play(false);
        this._voiceBuffersG.push(buffer);
        this.cleanupVSG();
    }
}


AudioManager.cleanupVSG = function() {
    for (const buffer of this._voiceBuffersG) {
        if (!buffer.isPlaying()) buffer.destroy();
    }
    this._voiceBuffersG = this._voiceBuffersG.filter(buffer => buffer.isPlaying());
}

AudioManager.stopVoiceSoundsG = function(allQclear) {
    for (const buffer of this._voiceBuffersG) {
        buffer.destroy();
    }
    this._voiceBuffersG = [];
    if(allQclear) $game_VSounds.allQclearG();
}

AudioManager.playStaticVSG = function(se) {
    if (se.name) {
        this.loadStaticVS(se);
        for (const buffer of this._staticBuffersVS) {
            if (buffer.name === se.name) {
                buffer.stop();
                this.updateSeParameters(buffer, se);
                buffer.play(false);
                break;
            }
        }
    }
}

AudioManager.loadStaticVSG = function(se) {
    if (se.name && !this.isStaticVSG(se)) {
        const buffer = this.createBuffer("voice/", se.name);
        this._staticBuffersVSG.push(buffer);
    }
}

AudioManager.isStaticVSG = function(se) {
    for (const buffer of this._staticBuffersVSG) {
        if (buffer.name === se.name) {
            return true;
        }
    }
    return false;
}

const SoR_VSM_AM_stopAll = AudioManager.stopAll;
AudioManager.stopAll = function() {
    SoR_VSM_AM_stopAll.call(this);
    this.stopVoiceSounds(false);
    this.stopVoiceSoundsG(false);
}

})();


///////////////////////////////////////////////////////////////////////////////////
function Game_VSoundSoR() { this.initialize(...arguments); }
Game_VSoundSoR.prototype.initialize = function() {
    this.initManager();
    this._queueProcessed = null;
    this.IsQueueSuspended = false;
    this._queueProcessedG = null;
    this.IsQueueSuspendedG = false;
}

Game_VSoundSoR.prototype.initManager = function() {
    this.SoRVSM_manager = {
        q_sounds: [],
        q_soundsG: []
    };
}

Game_VSoundSoR.prototype.addQsound = function(qs) {
    this.SoRVSM_manager.q_sounds = [];
    for(const q of qs) this.SoRVSM_manager.q_sounds.push(q);
}

Game_VSoundSoR.prototype.addQsoundG = function(qs) {
    this.SoRVSM_manager.q_soundsG = [];
    for(const q of qs) this.SoRVSM_manager.q_soundsG.push(q);
}

Game_VSoundSoR.prototype.addQrestored = function() {
    this.SoRVSM_manager.q_sounds.unshift(this._queueProcessed);
    this.processVoiceSounds();
}

Game_VSoundSoR.prototype.addQrestoredG = function() {
    this.SoRVSM_manager.q_soundsG.unshift(this._queueProcessedG);
    this.processVoiceSoundsG();
}

Game_VSoundSoR.prototype.allQclear = function() {
    this.SoRVSM_manager.q_sounds = [];
    this._queueProcessed = null;
}

Game_VSoundSoR.prototype.allQclearG = function() {
    this.SoRVSM_manager.q_soundsG = [];
    this._queueProcessedG = null;
}

Game_VSoundSoR.prototype.popQsound = function() {
    if(this.SoRVSM_manager.q_sounds.length==0) return null;
    const q = this.SoRVSM_manager.q_sounds.shift();
    return q;
}

Game_VSoundSoR.prototype.popQsoundG = function() {
    if(this.SoRVSM_manager.q_soundsG.length==0) return null;
    const q = this.SoRVSM_manager.q_soundsG.shift();
    return q;
}

Game_VSoundSoR.prototype.hasQsound = function() {
    return this.SoRVSM_manager.q_sounds.length>0;
}

Game_VSoundSoR.prototype.hasQsoundG = function() {
    return this.SoRVSM_manager.q_soundsG.length>0;
}

Game_VSoundSoR.prototype.processSoundQueue = function() {
    if(this.IsQueueSuspended){ // due to message system on RPGMAKER between two scenes
        this.IsQueueSuspended = false;
        return;
    }

    const q = this.popQsound();
    if(q==null) return null;
    AudioManager.stopVoiceSounds();
    AudioManager.playVoiceSound(q);
    return q;
}

Game_VSoundSoR.prototype.processSoundQueueG = function() {
    if(this.IsQueueSuspendedG){ // due to message system on RPGMAKER between two scenes
        this.IsQueueSuspendedG = false;
        return;
    }

    const q = this.popQsoundG();
    if(q==null) return null;
    AudioManager.stopVoiceSoundsG();
    AudioManager.playVoiceSoundG(q);
    return q;
}

Game_VSoundSoR.prototype.processVoiceSounds = function() {
    const ret = this.processSoundQueue();
    this._queueProcessed = ret;
}

Game_VSoundSoR.prototype.processVoiceSoundsG = function() {
    const ret = this.processSoundQueueG();
    this._queueProcessedG = ret;
}

//suspend mechanism is temporary.
Game_VSoundSoR.prototype.suspendVoiceSounds = function() {
    if(this._queueProcessed){
        this.IsQueueSuspended = true;
        AudioManager.stopVoiceSounds(false);
    }
}

Game_VSoundSoR.prototype.suspendVoiceSoundsG = function() {
    if(this._queueProcessedG){
        this.IsQueueSuspendedG = true;
        AudioManager.stopVoiceSoundsG(false);
    }
}

Game_VSoundSoR.prototype.isPlaying = function() {
    return AudioManager._voiceBuffers.some(x=> x.isPlaying());
}

Game_VSoundSoR.prototype.isPlayingG = function() {
    return AudioManager._voiceBuffersG.some(x=> x.isPlaying());
}